package com.planet_ink.coffee_mud.utils;
import java.util.*;
import com.planet_ink.coffee_mud.interfaces.*;
import com.planet_ink.coffee_mud.common.*;

/* 
   Copyright 2000-2005 Bo Zimmerman

   Licensed under the Apache License, Version 2.0 (the "License");
   you may not use this file except in compliance with the License.
   You may obtain a copy of the License at

       http://www.apache.org/licenses/LICENSE-2.0

   Unless required by applicable law or agreed to in writing, software
   distributed under the License is distributed on an "AS IS" BASIS,
   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
   See the License for the specific language governing permissions and
   limitations under the License.
*/
public class MUDZapper
{
	private MUDZapper(){}
	
	private static Hashtable zapCodes=new Hashtable();

	private static Hashtable getZapCodes()
	{
		if(zapCodes.size()==0)
		{
			zapCodes.put("-CLASS",new Integer(0));
			zapCodes.put("-CLASSES",new Integer(0));
			zapCodes.put("-BASECLASS",new Integer(1));
			zapCodes.put("-BASECLASSES",new Integer(1));
			zapCodes.put("-RACE",new Integer(2));
			zapCodes.put("-RACES",new Integer(2));
			zapCodes.put("-ALIGNMENT",new Integer(3));
			zapCodes.put("-ALIGNMENTS",new Integer(3));
			zapCodes.put("-ALIGN",new Integer(3));
			zapCodes.put("-GENDER",new Integer(4));
			zapCodes.put("-GENDERS",new Integer(4));
			zapCodes.put("-LEVEL",new Integer(5));
			zapCodes.put("-LEVELS",new Integer(5));
			zapCodes.put("-CLASSLEVEL",new Integer(6));
			zapCodes.put("-CLASSLEVELS",new Integer(6));
			zapCodes.put("-TATTOOS",new Integer(7));
			zapCodes.put("-TATTOO",new Integer(7));
			zapCodes.put("+TATTOOS",new Integer(8));
			zapCodes.put("+TATTOO",new Integer(8));
			zapCodes.put("-NAME",new Integer(9));
			zapCodes.put("-NAMES",new Integer(9));
			zapCodes.put("-PLAYER",new Integer(10));
			zapCodes.put("-NPC",new Integer(11));
			zapCodes.put("-MOB",new Integer(11));
			zapCodes.put("-RACECAT",new Integer(12));
			zapCodes.put("-RACECATS",new Integer(12));
			zapCodes.put("+RACECAT",new Integer(13));
			zapCodes.put("+RACECATS",new Integer(13));
			zapCodes.put("-CLAN",new Integer(14));
			zapCodes.put("-CLANS",new Integer(14));
			zapCodes.put("+CLAN",new Integer(15));
			zapCodes.put("+CLANS",new Integer(15));
			zapCodes.put("+NAME",new Integer(16));
			zapCodes.put("+NAMES",new Integer(16));
			zapCodes.put("-ANYCLASS",new Integer(17));
			zapCodes.put("+ANYCLASS",new Integer(18));
			zapCodes.put("+STR",new Integer(19));
			zapCodes.put("+INT",new Integer(20));
			zapCodes.put("+WIS",new Integer(21));
			zapCodes.put("+DEX",new Integer(22));
			zapCodes.put("+CON",new Integer(23));
			zapCodes.put("+CHA",new Integer(24));
			zapCodes.put("+STRENGTH",new Integer(19));
			zapCodes.put("+INTELLIGENCE",new Integer(20));
			zapCodes.put("+WISDOM",new Integer(21));
			zapCodes.put("+DEXTERITY",new Integer(22));
			zapCodes.put("+CONSTITUTION",new Integer(23));
			zapCodes.put("+CHARISMA",new Integer(24));
			zapCodes.put("-STR",new Integer(25));
			zapCodes.put("-STRENGTH",new Integer(25));
			zapCodes.put("-INT",new Integer(26));
			zapCodes.put("-INTELLIGENCE",new Integer(26));
			zapCodes.put("-WIS",new Integer(27));
			zapCodes.put("-WISDOM",new Integer(27));
			zapCodes.put("-DEX",new Integer(28));
			zapCodes.put("-DEXTERITY",new Integer(28));
			zapCodes.put("-CON",new Integer(29));
			zapCodes.put("-CONSTITUTION",new Integer(29));
			zapCodes.put("-CHA",new Integer(30));
			zapCodes.put("-CHARISMA",new Integer(30));
			zapCodes.put("-AREA",new Integer(31));
			zapCodes.put("+AREA",new Integer(32));
			zapCodes.put("+ITEM",new Integer(33));
			zapCodes.put("+CLASS",new Integer(34));  // for compiled use ONLY
			zapCodes.put("+ALIGNMENT",new Integer(35));  // for compiled use ONLY
			zapCodes.put("+GENDER",new Integer(36));  // for compiled use ONLY
			zapCodes.put("+LVLGR",new Integer(37));  // for compiled use ONLY
			zapCodes.put("+LVLGE",new Integer(38));  // for compiled use ONLY
			zapCodes.put("+LVLLT",new Integer(39));  // for compiled use ONLY
			zapCodes.put("+LVLLE",new Integer(40));  // for compiled use ONLY
			zapCodes.put("+LVLEQ",new Integer(41));  // for compiled use ONLY
			zapCodes.put("+EFFECTS",new Integer(42));
			zapCodes.put("+EFFECT",new Integer(42));
			zapCodes.put("-EFFECTS",new Integer(43));
			zapCodes.put("-EFFECT",new Integer(43));
			zapCodes.put("-DEITY",new Integer(44));
			zapCodes.put("+DEITY",new Integer(45));
			zapCodes.put("-FACTION",new Integer(46));
			zapCodes.put("+FACTION",new Integer(47));
		}
		return zapCodes;
	}

	private static final String ZAP ="+SYSOP (allow archons or area staff to bypass the rules)  <BR>"
									+"-SYSOP (always <WORD> archons and area staff)  <BR>"
									+"-PLAYER (<WORD> all players) <BR>"
									+"-MOB (<WORD> all mobs/npcs)  <BR>"
									+"-CLASS  (<WORD> all classes)  <BR>"
									+"-BASECLASS  (<WORD> all base classes)  <BR>"
									+"+thief +mage +ranger (create exceptions to -class and -baseclass) <BR>"
									+"-thief -mage  -ranger (<WORD> only listed classes)<BR>"
									+"-RACE (<WORD> all races)  <BR>"
									+"+elf +dwarf +human +half +gnome (create exceptions to -race)  <BR>"
									+"-elf -dwarf -human -half -gnome (<WORD> only listed races)  <BR>"
									+"-RACECAT (<WORD> all racial categories)  <BR>"
									+"+RACECAT (do not <WORD> all racial categories)  <BR>"
									+"+elf +insect +humanoid +canine +gnome (create exceptions to -racecat)  <BR>"
									+"-elf -insect -humanoid -canine -gnome (create exceptions to +racecat)  <BR>"
									+"-ALIGNMENT (<WORD> all alignments)  <BR>"
									+"+evil +good +neutral (create exceptions to -alignment)  <BR>"
									+"-evil -good -neutral (<WORD> only listed alignments)  <BR>"
									+"-GENDER (<WORD> all genders)  <BR>"
									+"+male +female +neuter (create exceptions to -gender)  <BR>"
									+"-male -female -neuter (<WORD> only listed genders)  <BR>"
									+"-FACTION (<WORD> all faction and values, even a lack of faction) <BR>"
									+"+myfactionrange +myotherfactionrange (create exceptions to -faction) <BR>"
									+"-myfactionrange -myotherfactionrange (<WORD> only named faction range)<BR>"
									+"-TATTOOS (<WORD> all tattoos, even a lack of a tatoo) <BR>"
									+"+mytatto +thistattoo +anytattoo etc..  (create exceptions to -tattoos) <BR>"
									+"+TATTOOS (do not <WORD> any or no tattoos) <BR>"
									+"-mytattoo -anytatto, etc.. (create exceptions to +tattoos) <BR>"
									+"-LEVEL (<WORD> all levels)  <BR>"
									+"+=1 +>5 +>=7 +<13 +<=20 (create exceptions to -level using level ranges)  <BR>"
									+"-=1 ->5 ->=7 -<13 -<=20 (<WORD> only listed levels range) <BR>"
									+"-NAMES (<WORD> everyone) <BR>"
									+"+bob \"+my name\" etc.. (create name exceptions to -names) <BR>"
									+"+NAMES (do not <WORD> anyone who has a name) <BR>"
									+"-bob \"-my name\" etc.. (create name exceptions to +names) <BR>"
									+"-CLAN (<WORD> anyone, even no clan) <BR>"
									+"+Killers \"+Holy Avengers\" etc.. (create clan exceptions to -clan) <BR>"
									+"+CLAN (do not <WORD> anyone, even non clan people) <BR>"
									+"-Killers \"-Holy Avengers\" etc.. (create clan exceptions to +clan) <BR>"
									+"-DEITY (<WORD> anyone, even no deity) <BR>"
									+"+Apollo \"+Grothon The Great\" etc.. (create deity exceptions to -deity) <BR>"
									+"+DEITY (do not <WORD> anyone, even non deity worshipping people) <BR>"
									+"-Apollo \"-rothon The Great\" etc.. (create deity exceptions to +deity) <BR>"
									+"-ANYCLASS (<WORD> all multi-class combinations)  <BR>"
									+"+thief +mage +ranger (exceptions -anyclass, allow any levels) <BR>"
									+"+ANYCLASS (do not <WORD> all multi-class combinations)  <BR>"
									+"-thief -mage -ranger (exceptions to +anyclass, disallow any levels) <BR>"
									+"-STR X (<WORD> those with strength less than X)  <BR>"
									+"+STR X (<WORD> those with strength greater than X)  <BR>"
									+"-INT X (<WORD> those with intelligence less than X)  <BR>"
									+"+INT X (<WORD> those with intelligence greater than X)  <BR>"
									+"-WIS X (<WORD> those with wisdom less than X)  <BR>"
									+"+WIS X (<WORD> those with wisdom greater than X)  <BR>"
									+"-CON X (<WORD> those with constitution less than X)  <BR>"
									+"+CON X (<WORD> those with constitution greater than X)  <BR>"
									+"-CHA X (<WORD> those with charisma less than X)  <BR>"
									+"+CHA X (<WORD> those with charisma greater than X)  <BR>"
									+"-DEX X (<WORD> those with dexterity less than X)  <BR>"
									+"+DEX X (<WORD> those with dexterity greater than X) <BR>"
									+"-AREA (<WORD> in all areas) <BR>"
									+"\"+my areaname\" etc.. (create exceptions to +area) <BR>"
									+"+AREA (do not <WORD> any areas) <BR>"
									+"\"-my areaname\" etc.. (create exceptions to -area) <BR>"
									+"+ITEM \"+item name\" etc... (<WORD> only those with an item name) <BR>"
									+"-EFFECTS (<WORD> anyone, even no effects) <BR>"
									+"+Sleep \"+Wood Chopping\" etc.. (create name exceptions to -effects) <BR>"
									+"+EFFECTS (do not <WORD> anyone, even non effected people) <BR>"
									+"-Sleep \"-Wood Chopping\" etc.. (create name exceptions to +effects) ";

	public static String zapperInstructions(String CR, String word)
	{
		String copy=new String(ZAP);
		if((CR!=null)&&(!CR.equalsIgnoreCase("<BR>")))
			copy=Util.replaceAll(copy,"<BR>",CR);
		if((word==null)||(word.length()==0))
			copy=Util.replaceAll(copy,"<WORD>","disallow");
		else
			copy=Util.replaceAll(copy,"<WORD>",word);
		return copy;
	}


	public static boolean tattooCheck(Vector V, char plusMinus, int fromHere, MOB mob)
	{
		for(int v=0;v<mob.numTattoos();v++)
		{
			String tattoo=mob.fetchTattoo(v);
			if((tattoo!=null)
			&&(tattoo.length()>0)
			&&(Character.isDigit(tattoo.charAt(0)))
			&&(tattoo.indexOf(" ")>0)
			&&(Util.isNumber(tattoo.substring(0,tattoo.indexOf(" ")))))
			   tattoo=tattoo.substring(tattoo.indexOf(" ")+1).trim();
			if(fromHere(V,plusMinus,fromHere,tattoo))
				return true;
		}
		return false;
	}

	private static boolean levelCheck(String text, char prevChar, int lastPlace, int lvl)
	{
		int x=0;
		while(x>=0)
		{
			x=text.indexOf(">",lastPlace);
			if(x<0)	x=text.indexOf("<",lastPlace);
			if(x<0)	x=text.indexOf("=",lastPlace);
			if(x>=0)
			{
				char prev='+';
				if(x>0) prev=text.charAt(x-1);

				char primaryChar=text.charAt(x);
				x++;
				boolean andEqual=false;
				if(text.charAt(x)=='=')
				{
					andEqual=true;
					x++;
				}
				lastPlace=x;

				if(prev==prevChar)
				{
					boolean found=false;
					String cmpString="";
					while((x<text.length())&&
						  (((text.charAt(x)==' ')&&(cmpString.length()==0))
						   ||(Character.isDigit(text.charAt(x)))))
					{
						if(Character.isDigit(text.charAt(x)))
							cmpString+=text.charAt(x);
						x++;
					}
					if(cmpString.length()>0)
					{
						int cmpLevel=Util.s_int(cmpString);
						if((cmpLevel==lvl)&&(andEqual))
							found=true;
						else
						switch(primaryChar)
						{
						case '>': found=(lvl>cmpLevel); break;
						case '<': found=(lvl<cmpLevel); break;
						case '=': found=(lvl==cmpLevel); break;
						}
					}
					if(found) return true;
				}
			}
		}
		return false;
	}

	public static Vector levelCompiledHelper(String str, char c, Vector entry)
	{
		if(entry==null) entry=new Vector();
		if(str.startsWith(c+">="))
		{
			entry.addElement(zapCodes.get("LVLGE"));
			entry.addElement(new Integer(Util.s_int(str.substring(3).trim())));
		}
		else
		if(str.startsWith(c+"<="))
		{
			entry.addElement(zapCodes.get("LVLLE"));
			entry.addElement(new Integer(Util.s_int(str.substring(3).trim())));
		}
		else
		if(str.startsWith(c+">"))
		{
			entry.addElement(zapCodes.get("LVLGT"));
			entry.addElement(new Integer(Util.s_int(str.substring(3).trim())));
		}
		else
		if(str.startsWith(c+"<"))
		{
			entry.addElement(zapCodes.get("LVLLT"));
			entry.addElement(new Integer(Util.s_int(str.substring(3).trim())));
		}
		else
		if(str.startsWith(c+"="))
		{
			entry.addElement(zapCodes.get("LVLEQ"));
			entry.addElement(new Integer(Util.s_int(str.substring(3).trim())));
		}
		return entry;
	}
	
	public static StringBuffer levelHelp(String str, char c, String append)
	{
		if(str.startsWith(c+">="))
			return new StringBuffer(append+"levels greater than or equal to "+str.substring(3).trim()+".  ");
		else
		if(str.startsWith(c+"<="))
			return new StringBuffer(append+"levels less than or equal to "+str.substring(3).trim()+".  ");
		else
		if(str.startsWith(c+">"))
			return new StringBuffer(append+"levels greater than "+str.substring(2).trim()+".  ");
		else
		if(str.startsWith(c+"<"))
			return new StringBuffer(append+"levels less than "+str.substring(2).trim()+".  ");
		else
		if(str.startsWith(c+"="))
			return new StringBuffer(append+"level "+str.substring(2).trim()+" players.  ");
		return new StringBuffer("");
	}
	
	
	public static boolean fromHereEqual(Vector V, char plusMinus, int fromHere, String find)
	{
		for(int v=fromHere;v<V.size();v++)
		{
			String str=(String)V.elementAt(v);
			if(str.length()==0) continue;
			if(zapCodes.containsKey(str))
				return false;
			if(str.equalsIgnoreCase(plusMinus+find)) return true;
		}
		return false;
	}

	public static boolean factionCheck(Vector V, char plusMinus, int fromHere, MOB mob)
	{
		for(int v=fromHere;v<V.size();v++)
		{
			String str=((String)V.elementAt(v)).toUpperCase();
			if(str.length()>0)
			{
				if(zapCodes.containsKey(str))
					return false;
				if((str.charAt(0)==plusMinus)
				&&(Factions.isFactionedThisWay(mob,str.substring(1))))
				    return true;
			}
		}
		return false;
	}

	public static boolean nameCheck(Vector V, char plusMinus, int fromHere, MOB mob)
	{
		Vector names=Util.parse(mob.name().toUpperCase());
		for(int v=0;v<names.size();v++)
			if(fromHereEqual(V,plusMinus,fromHere,(String)names.elementAt(v)))
				return true;
		names=Util.parse(mob.displayText().toUpperCase());
		for(int v=0;v<names.size();v++)
			if(fromHereEqual(V,plusMinus,fromHere,(String)names.elementAt(v)))
				return true;
		return false;
	}

	public static boolean areaCheck(Vector V, char plusMinus, int fromHere, MOB mob)
	{
		if((mob==null)||(mob.location()==null)) return false;
		Area A=mob.location().getArea();
		if(A==null) return false;
		return fromHere(V,plusMinus,fromHere,A.name());
	}

	public static boolean itemCheck(Vector V, char plusMinus, int fromHere, MOB mob)
	{
		if((mob==null)||(mob.location()==null)) return false;
		for(int v=fromHere;v<V.size();v++)
		{
			String str=(String)V.elementAt(v);
			if(str.length()==0) continue;
			if(zapCodes.containsKey(str))
				return false;
			if(mob.fetchInventory(str)!=null)
				return true;
		}
		return false;
	}
	
	public static boolean fromHere(Vector V, char plusMinus, int fromHere, String find)
	{
		for(int v=fromHere;v<V.size();v++)
		{
			String str=(String)V.elementAt(v);
			if(str.length()==0) continue;
			if(zapCodes.containsKey(str))
				return false;
			if(str.startsWith(plusMinus+find)) return true;
		}
		return false;
	}

	public static String zapperDesc(String text)
	{
		if(text.trim().length()==0) return "Anyone";
		StringBuffer buf=new StringBuffer("");
		getZapCodes();
		Vector V=Util.parse(text.toUpperCase());
		for(int v=0;v<V.size();v++)
		{
			String str=(String)V.elementAt(v);
			int val=-1;
			if(zapCodes.containsKey(str))
				switch(((Integer)zapCodes.get(str)).intValue())
				{
				case 0: // -class
					{
					buf.append("Allows only ");
					for(Enumeration c=CMClass.charClasses();c.hasMoreElements();)
					{
						CharClass C=(CharClass)c.nextElement();
						if(fromHere(V,'+',v+1,Util.padRight(C.name(),4).toUpperCase().trim()))
							buf.append(C.name()+", ");
					}
					if(buf.toString().endsWith(", "))
						buf=new StringBuffer(buf.substring(0,buf.length()-2));
					buf.append(".  ");
					}
					break;
				case 1: // -baseclass
					{
						buf.append("Allows only ");
						for(Enumeration c=CMClass.charClasses();c.hasMoreElements();)
						{
							CharClass C=(CharClass)c.nextElement();
							if((C.ID().equals(C.baseClass()))
							&&(fromHere(V,'+',v+1,Util.padRight(C.name(),4).toUpperCase().trim())))
								buf.append(C.name()+" types, ");
						}
						if(buf.toString().endsWith(", "))
							buf=new StringBuffer(buf.substring(0,buf.length()-2));
						buf.append(".  ");
					}
					break;
				case 2: // -Race
					{
						buf.append("Allows only ");
						Vector cats=new Vector();
						for(Enumeration r=CMClass.races();r.hasMoreElements();)
						{
							Race R=(Race)r.nextElement();
							String cat=R.name().toUpperCase();
							if(cat.length()>6) cat=cat.substring(0,6);
							if((!cats.contains(R.name())
							&&(fromHere(V,'+',v+1,cat))))
							   cats.addElement(R.name());
						}
						for(int c=0;c<cats.size();c++)
							buf.append(((String)cats.elementAt(c))+", ");
						if(buf.toString().endsWith(", "))
							buf=new StringBuffer(buf.substring(0,buf.length()-2));
						buf.append(".  ");
					}
					break;
				case 12: // -Racecats
					{
						buf.append("Allows only these racial categories ");
						Vector cats=new Vector();
						for(Enumeration r=CMClass.races();r.hasMoreElements();)
						{
							Race R=(Race)r.nextElement();
							String cat=R.racialCategory().toUpperCase();
							if((!cats.contains(R.racialCategory())
							&&(fromHere(V,'+',v+1,cat))))
							   cats.addElement(R.racialCategory());
						}
						for(int c=0;c<cats.size();c++)
							buf.append(((String)cats.elementAt(c))+", ");
						if(buf.toString().endsWith(", "))
							buf=new StringBuffer(buf.substring(0,buf.length()-2));
						buf.append(".  ");
					}
					break;
				case 3: // -Alignment
					{
						buf.append("Allows only ");
						if(fromHere(V,'+',v+1,Faction.ALIGN_NAMES[Faction.ALIGN_EVIL].substring(0,3)))
							buf.append(Faction.ALIGN_NAMES[Faction.ALIGN_EVIL].toLowerCase()+", ");
						if(fromHere(V,'+',v+1,Faction.ALIGN_NAMES[Faction.ALIGN_GOOD].substring(0,3)))
							buf.append(Faction.ALIGN_NAMES[Faction.ALIGN_GOOD].toLowerCase()+", ");
						if(fromHere(V,'+',v+1,Faction.ALIGN_NAMES[Faction.ALIGN_NEUTRAL].substring(0,3)))
							buf.append(Faction.ALIGN_NAMES[Faction.ALIGN_NEUTRAL].toLowerCase()+", ");
						if(buf.toString().endsWith(", "))
							buf=new StringBuffer(buf.substring(0,buf.length()-2));
						buf.append(".  ");
					}
					break;
				case 4: // -Gender
					{
						buf.append("Allows only ");
						if(fromHere(V,'+',v+1,"MALE"))
							buf.append("Male, ");
						if(fromHere(V,'+',v+1,"FEMALE"))
							buf.append("Female, ");
						if(fromHere(V,'+',v+1,"NEUTER"))
							buf.append("Neuter");
						if(buf.toString().endsWith(", "))
							buf=new StringBuffer(buf.substring(0,buf.length()-2));
						buf.append(".  ");
					}
					break;
				case 5: // -Levels
					{
						for(int v2=v+1;v2<V.size();v2++)
							buf.append(levelHelp((String)V.elementAt(v2),'+',"Allows only "));
					}
					break;
				case 6: // -ClassLevels
					{
						for(int v2=v+1;v2<V.size();v2++)
							buf.append(levelHelp((String)V.elementAt(v2),'+',"Allows only class "));
					}
					break;
				case 7: // -Tattoos
					{
						buf.append("Requires the following tattoo(s): ");
						for(int v2=v+1;v2<V.size();v2++)
						{
							String str2=(String)V.elementAt(v2);
							if((!zapCodes.containsKey(str2))&&(str2.startsWith("+")))
								buf.append(str2+", ");
						}
						if(buf.toString().endsWith(", "))
							buf=new StringBuffer(buf.substring(0,buf.length()-2));
						buf.append(".  ");
					}
					break;
				case 8: // +Tattoos
					{
						buf.append("Disallows the following tattoo(s): ");
						for(int v2=v+1;v2<V.size();v2++)
						{
							String str2=(String)V.elementAt(v2);
							if((!zapCodes.containsKey(str2))&&(str2.startsWith("-")))
								buf.append(str2+", ");
						}
						if(buf.toString().endsWith(", "))
							buf=new StringBuffer(buf.substring(0,buf.length()-2));
						buf.append(".  ");
					}
					break;
				case 14: // -Clan
					{
						buf.append("Requires membership in the following clan(s): ");
						for(int v2=v+1;v2<V.size();v2++)
						{
							String str2=(String)V.elementAt(v2);
							if((!zapCodes.containsKey(str2))&&(str2.startsWith("+")))
								buf.append(str2+", ");
						}
						if(buf.toString().endsWith(", "))
							buf=new StringBuffer(buf.substring(0,buf.length()-2));
						buf.append(".  ");
					}
					break;
				case 15: // +Clan
					{
						buf.append("Disallows the following clan(s): ");
						for(int v2=v+1;v2<V.size();v2++)
						{
							String str2=(String)V.elementAt(v2);
							if((!zapCodes.containsKey(str2))&&(str2.startsWith("-")))
								buf.append(str2+", ");
						}
						if(buf.toString().endsWith(", "))
							buf=new StringBuffer(buf.substring(0,buf.length()-2));
						buf.append(".  ");
					}
					break;
				case 44: // -Deity
					{
						buf.append("Requires worshipping in the following deity(s): ");
						for(int v2=v+1;v2<V.size();v2++)
						{
							String str2=(String)V.elementAt(v2);
							if((!zapCodes.containsKey(str2))&&(str2.startsWith("+")))
								buf.append(str2+", ");
						}
						if(buf.toString().endsWith(", "))
							buf=new StringBuffer(buf.substring(0,buf.length()-2));
						buf.append(".  ");
					}
					break;
				case 45: // +Deity
					{
						buf.append("Disallows the worshippers of: ");
						for(int v2=v+1;v2<V.size();v2++)
						{
							String str2=(String)V.elementAt(v2);
							if((!zapCodes.containsKey(str2))&&(str2.startsWith("-")))
								buf.append(str2+", ");
						}
						if(buf.toString().endsWith(", "))
							buf=new StringBuffer(buf.substring(0,buf.length()-2));
						buf.append(".  ");
					}
					break;
				case 16: // +Names
					{
						buf.append("Disallows the following mob/player name(s): ");
						for(int v2=v+1;v2<V.size();v2++)
						{
							String str2=(String)V.elementAt(v2);
							if((!zapCodes.containsKey(str2))&&(str2.startsWith("-")))
								buf.append(str2+", ");
						}
						if(buf.toString().endsWith(", "))
							buf=new StringBuffer(buf.substring(0,buf.length()-2));
						buf.append(".  ");
					}
					break;
				case 9: // -Names
					{
						buf.append("Requires the following name(s): ");
						for(int v2=v+1;v2<V.size();v2++)
						{
							String str2=(String)V.elementAt(v2);
							if((!zapCodes.containsKey(str2))&&(str2.startsWith("+")))
								buf.append(str2+", ");
						}
						if(buf.toString().endsWith(", "))
							buf=new StringBuffer(buf.substring(0,buf.length()-2));
						buf.append(".  ");
					}
					break;
				case 10: // -Player
					buf.append("Disallows players.  ");
					break;
				case 11: // -MOB
					buf.append("Disallows mobs/npcs.  ");
					break;
				case 13: // +racecats
					{
						buf.append("Disallows the following racial cat(s): ");
						for(int v2=v+1;v2<V.size();v2++)
						{
							String str2=(String)V.elementAt(v2);
							if((!zapCodes.containsKey(str2))&&(str2.startsWith("-")))
								buf.append(str2+", ");
						}
						if(buf.toString().endsWith(", "))
							buf=new StringBuffer(buf.substring(0,buf.length()-2));
						buf.append(".  ");
					}
					break;
				case 17: // -anyclass
					{
						buf.append("Requires levels in one of the following:  ");
						for(Enumeration c=CMClass.charClasses();c.hasMoreElements();)
						{
							CharClass C=(CharClass)c.nextElement();
							if(fromHere(V,'+',v+1,Util.padRight(C.name(),4).toUpperCase().trim()))
								buf.append(C.name()+", ");
						}
						if(buf.toString().endsWith(", "))
							buf=new StringBuffer(buf.substring(0,buf.length()-2));
						buf.append(".  ");
					}
					break;
				case 18: // +anyclass
					{
						buf.append("Disallows any levels in any of the following:  ");
						for(Enumeration c=CMClass.charClasses();c.hasMoreElements();)
						{
							CharClass C=(CharClass)c.nextElement();
							if(fromHere(V,'-',v+1,Util.padRight(C.name(),4).toUpperCase().trim()))
								buf.append(C.name()+", ");
						}
						if(buf.toString().endsWith(", "))
							buf=new StringBuffer(buf.substring(0,buf.length()-2));
						buf.append(".  ");
					}
					break;
				case 19: // -str
					val=((++v)<V.size())?Util.s_int((String)V.elementAt(v)):0;
					buf.append("Requires a strength of at least "+val+".");
					break;
				case 20: // -int
					val=((++v)<V.size())?Util.s_int((String)V.elementAt(v)):0;
					buf.append("Requires a intelligence of at least "+val+".");
					break;
				case 21: // -wis
					val=((++v)<V.size())?Util.s_int((String)V.elementAt(v)):0;
					buf.append("Requires a wisdom of at least "+val+".");
					break;
				case 22: // -dex
					val=((++v)<V.size())?Util.s_int((String)V.elementAt(v)):0;
					buf.append("Requires a dexterity of at least "+val+".");
					break;
				case 23: // -con
					val=((++v)<V.size())?Util.s_int((String)V.elementAt(v)):0;
					buf.append("Requires a constitution of at least "+val+".");
					break;
				case 24: // -cha
					val=((++v)<V.size())?Util.s_int((String)V.elementAt(v)):0;
					buf.append("Requires a charisma of at least "+val+".");
					break;
				case 25: // +str
					val=((++v)<V.size())?Util.s_int((String)V.elementAt(v)):0;
					buf.append("Requires a strength of at most "+val+".");
					break;
				case 26: // +int
					val=((++v)<V.size())?Util.s_int((String)V.elementAt(v)):0;
					buf.append("Requires a intelligence of at most "+val+".");
					break;
				case 27: // +wis
					val=((++v)<V.size())?Util.s_int((String)V.elementAt(v)):0;
					buf.append("Requires a wisdom of at most "+val+".");
					break;
				case 28: // +dex
					val=((++v)<V.size())?Util.s_int((String)V.elementAt(v)):0;
					buf.append("Requires a dexterity of at most "+val+".");
					break;
				case 29: // +con
					val=((++v)<V.size())?Util.s_int((String)V.elementAt(v)):0;
					buf.append("Requires a constitution of at most "+val+".");
					break;
				case 30: // +cha
					val=((++v)<V.size())?Util.s_int((String)V.elementAt(v)):0;
					buf.append("Requires a charisma of at most "+val+".");
					break;
				case 31: // +Area
					{
						buf.append("Disallows the following area(s): ");
						for(int v2=v+1;v2<V.size();v2++)
						{
							String str2=(String)V.elementAt(v2);
							if((!zapCodes.containsKey(str2))&&(str2.startsWith("-")))
								buf.append(str2+", ");
						}
						if(buf.toString().endsWith(", "))
							buf=new StringBuffer(buf.substring(0,buf.length()-2));
						buf.append(".  ");
					}
					break;
				case 32: // -Area
					{
						buf.append("Requires the following area(s): ");
						for(int v2=v+1;v2<V.size();v2++)
						{
							String str2=(String)V.elementAt(v2);
							if((!zapCodes.containsKey(str2))&&(str2.startsWith("+")))
								buf.append(str2+", ");
						}
						if(buf.toString().endsWith(", "))
							buf=new StringBuffer(buf.substring(0,buf.length()-2));
						buf.append(".  ");
					}
					break;
				case 33: // +Item
					{
						buf.append("Requires the following item(s): ");
						for(int v2=v+1;v2<V.size();v2++)
						{
							String str2=(String)V.elementAt(v2);
							if((!zapCodes.containsKey(str2))&&(str2.startsWith("-")))
								buf.append(str2+", ");
						}
						if(buf.toString().endsWith(", "))
							buf=new StringBuffer(buf.substring(0,buf.length()-2));
						buf.append(".  ");
					}
					break;
				case 42: // +Effects
					{
						buf.append("Disallows the following activities/effects(s): ");
						for(int v2=v+1;v2<V.size();v2++)
						{
							String str2=(String)V.elementAt(v2);
							if((!zapCodes.containsKey(str2))&&(str2.startsWith("-")))
								buf.append(str2+", ");
						}
						if(buf.toString().endsWith(", "))
							buf=new StringBuffer(buf.substring(0,buf.length()-2));
						buf.append(".  ");
					}
					break;
				case 43: // -Effects
					{
						buf.append("Requires participation in the following activities/effects(s): ");
						for(int v2=v+1;v2<V.size();v2++)
						{
							String str2=(String)V.elementAt(v2);
							if((!zapCodes.containsKey(str2))&&(str2.startsWith("+")))
								buf.append(str2+", ");
						}
						if(buf.toString().endsWith(", "))
							buf=new StringBuffer(buf.substring(0,buf.length()-2));
						buf.append(".  ");
					}
					break;
				case 46: // -faction
				{
				    buf.append("Requires the following: ");
				    for(int v2=v+1;v2<V.size();v2++)
				    {
				        String str2=(String)V.elementAt(v2);
				        if(zapCodes.containsKey(str2))
				            break;
				        if((str2.startsWith("+"))
				        &&(Factions.isRangeCodeName(str2.substring(1))))
				        {
				            String desc=Factions.rangeDescription(str2.substring(1),"or ");
				            if(desc.length()>0) buf.append(desc+"; ");
				        }
				    }
					if(buf.toString().endsWith(", "))
						buf=new StringBuffer(buf.substring(0,buf.length()-2));
					if(buf.toString().endsWith("; "))
						buf=new StringBuffer(buf.substring(0,buf.length()-2));
					buf.append(".  ");
				}
				break;
				}
			else
			{
				for(Enumeration c=CMClass.charClasses();c.hasMoreElements();)
				{
					CharClass C=(CharClass)c.nextElement();
					if(str.startsWith("-"+Util.padRight(C.name(),4).toUpperCase().trim()))
						buf.append("Disallows "+C.name()+".  ");
				}
				Vector cats=new Vector();
				for(Enumeration r=CMClass.races();r.hasMoreElements();)
				{
					Race R=(Race)r.nextElement();
					String cat=R.racialCategory().toUpperCase();
					if(cat.length()>6) cat=cat.substring(0,6);
					if((str.startsWith("-"+cat))&&(!cats.contains(R.racialCategory())))
					{
						cats.addElement(R.racialCategory());
						buf.append("Disallows "+R.racialCategory()+".  ");
					}
				}
				if(str.startsWith("-"+Faction.ALIGN_NAMES[Faction.ALIGN_EVIL].substring(0,3)))
					buf.append("Disallows "+Faction.ALIGN_NAMES[Faction.ALIGN_EVIL].toLowerCase()+".  ");
				if(str.startsWith("-"+Faction.ALIGN_NAMES[Faction.ALIGN_GOOD].substring(0,3)))
					buf.append("Disallows "+Faction.ALIGN_NAMES[Faction.ALIGN_GOOD].toLowerCase()+".  ");
				if(str.startsWith("-"+Faction.ALIGN_NAMES[Faction.ALIGN_NEUTRAL].substring(0,3)))
					buf.append("Disallows "+Faction.ALIGN_NAMES[Faction.ALIGN_NEUTRAL].toLowerCase()+".  ");
				if(str.startsWith("-MALE"))
					buf.append("Disallows Males.  ");
				if(str.startsWith("-FEMALE"))
					buf.append("Disallows Females.  ");
				if(str.startsWith("-NEUTER"))
					buf.append("Allows only Males and Females.  ");
				buf.append(levelHelp(str,'-',"Disallows "));
				if((str.startsWith("-"))
		        &&(Factions.isRangeCodeName(str.substring(1))))
				{
		            String desc=Factions.rangeDescription(str.substring(1),"and ");
		            if(desc.length()>0) buf.append("Disallows "+desc);
				}
			}
		}

		if(buf.length()==0) buf.append("Anyone.");
		return buf.toString();
	}

	public static Vector zapperCompile(String text)
	{
		Vector buf=new Vector();
		if(text.trim().length()==0) return buf;
		getZapCodes();
		Vector V=Util.parse(text.toUpperCase());
		for(int v=0;v<V.size();v++)
		{
			String str=(String)V.elementAt(v);
			int val=-1;
			if(zapCodes.containsKey(str))
				switch(((Integer)zapCodes.get(str)).intValue())
				{
				case 0: // -class
					{
						Vector entry=new Vector();
						buf.addElement(entry);
						entry.addElement(zapCodes.get(str));
						for(Enumeration c=CMClass.charClasses();c.hasMoreElements();)
						{
							CharClass C=(CharClass)c.nextElement();
							if(fromHere(V,'+',v+1,Util.padRight(C.name(),4).toUpperCase().trim()))
								entry.addElement(C.name());
						}
					}
					break;
				case 1: // -baseclass
					{
						Vector entry=new Vector();
						buf.addElement(entry);
						entry.addElement(zapCodes.get(str));
						for(Enumeration c=CMClass.charClasses();c.hasMoreElements();)
						{
							CharClass C=(CharClass)c.nextElement();
							if((C.ID().equals(C.baseClass())
							&&(fromHere(V,'+',v+1,Util.padRight(C.name(),4).toUpperCase().trim()))))
								entry.addElement(C.baseClass());
						}
					}
					break;
				case 2: // -Race
					{
						Vector entry=new Vector();
						buf.addElement(entry);
						entry.addElement(zapCodes.get(str));
						Vector cats=new Vector();
						for(Enumeration r=CMClass.races();r.hasMoreElements();)
						{
							Race R=(Race)r.nextElement();
							String cat=R.name().toUpperCase();
							if(cat.length()>6) cat=cat.substring(0,6);
							if((!cats.contains(R.name())
							&&(fromHere(V,'+',v+1,cat))))
							   cats.addElement(R.name());
						}
						for(int c=0;c<cats.size();c++)
							entry.addElement(cats.elementAt(c));
					}
					break;
				case 12: // -Racecats
					{
						Vector entry=new Vector();
						buf.addElement(entry);
						entry.addElement(zapCodes.get(str));
						Vector cats=new Vector();
						for(Enumeration r=CMClass.races();r.hasMoreElements();)
						{
							Race R=(Race)r.nextElement();
							String cat=R.racialCategory().toUpperCase();
							if((!cats.contains(R.racialCategory())
							&&(fromHere(V,'+',v+1,cat))))
							   cats.addElement(R.racialCategory());
						}
						for(int c=0;c<cats.size();c++)
							entry.addElement(cats.elementAt(c));
					}
					break;
				case 13: // +Racecats
					{
						Vector entry=new Vector();
						buf.addElement(entry);
						entry.addElement(zapCodes.get(str));
						Vector cats=new Vector();
						for(Enumeration r=CMClass.races();r.hasMoreElements();)
						{
							Race R=(Race)r.nextElement();
							String cat=R.racialCategory().toUpperCase();
							if((!cats.contains(R.racialCategory())
							&&(fromHere(V,'-',v+1,cat))))
							   cats.addElement(R.racialCategory());
						}
						for(int c=0;c<cats.size();c++)
							entry.addElement(cats.elementAt(c));
					}
					break;
				case 3: // -Alignment
					{
						Vector entry=new Vector();
						buf.addElement(entry);
						entry.addElement(zapCodes.get(str));
						if(fromHere(V,'+',v+1,Faction.ALIGN_NAMES[Faction.ALIGN_EVIL].substring(0,3)))
						    entry.addElement(Faction.ALIGN_NAMES[Faction.ALIGN_EVIL]);
						if(fromHere(V,'+',v+1,Faction.ALIGN_NAMES[Faction.ALIGN_GOOD].substring(0,3)))
						    entry.addElement(Faction.ALIGN_NAMES[Faction.ALIGN_GOOD]);
						if(fromHere(V,'+',v+1,Faction.ALIGN_NAMES[Faction.ALIGN_NEUTRAL].substring(0,3)))
						    entry.addElement(Faction.ALIGN_NAMES[Faction.ALIGN_NEUTRAL]);
					}
					break;
				case 4: // -Gender
					{
						Vector entry=new Vector();
						buf.addElement(entry);
						entry.addElement(zapCodes.get(str));
						if(fromHere(V,'+',v+1,"MALE"))
							entry.addElement("M");
						if(fromHere(V,'+',v+1,"FEMALE"))
							entry.addElement("F");
						if(fromHere(V,'+',v+1,"NEUTER"))
							entry.addElement("N");
					}
					break;
				case 5: // -Levels
				case 6: // -ClassLevels
					{
						Vector entry=new Vector();
						buf.addElement(entry);
						entry.addElement(zapCodes.get(str));
						for(int v2=v+1;v2<V.size();v2++)
							levelCompiledHelper((String)V.elementAt(v2),'+',entry);
					}
					break;
				case 7: // -Tattoos
				case 14: // -Clan
				case 44: // -Clan
				case 43: // -Effect
				case 9: // -Names
				case 32: // -Area
					{
						Vector entry=new Vector();
						buf.addElement(entry);
						entry.addElement(zapCodes.get(str));
						for(int v2=v+1;v2<V.size();v2++)
						{
							String str2=(String)V.elementAt(v2);
							if((!zapCodes.containsKey(str2))&&(str2.startsWith("+")))
								entry.addElement(str2);
						}
					}
					break;
				case 46: // -Faction
				{
					Vector entry=new Vector();
					buf.addElement(entry);
					entry.addElement(zapCodes.get(str));
					for(int v2=v+1;v2<V.size();v2++)
					{
						String str2=(String)V.elementAt(v2);
						if((!zapCodes.containsKey(str2))
						&&(str2.startsWith("+"))
				        &&(Factions.isRangeCodeName(str2.substring(1))))
							entry.addElement(str2.substring(1).toUpperCase());
					}
					break;
				}
				case 8: // +Tattoos
				case 15: // +Clan
				case 45: // +Clan
				case 42: // +Effect
				case 16: // +Names
				case 31: // +Area
				case 33: // +Item
					{
						Vector entry=new Vector();
						buf.addElement(entry);
						entry.addElement(zapCodes.get(str));
						for(int v2=v+1;v2<V.size();v2++)
						{
							String str2=(String)V.elementAt(v2);
							if((!zapCodes.containsKey(str2))&&(str2.startsWith("-")))
								entry.addElement(str2);
						}
					}
					break;
				case 10: // -Player
				case 11: // -MOB
					{
						Vector entry=new Vector();
						buf.addElement(entry);
						entry.addElement(zapCodes.get(str));
						break;
					}
				case 17: // -anyclass
					{
						Vector entry=new Vector();
						buf.addElement(entry);
						entry.addElement(zapCodes.get(str));
						for(Enumeration c=CMClass.charClasses();c.hasMoreElements();)
						{
							CharClass C=(CharClass)c.nextElement();
							if(fromHere(V,'+',v+1,Util.padRight(C.name(),4).toUpperCase().trim()))
								entry.addElement(C.name());
						}
					}
					break;
				case 18: // +anyclass
					{
						Vector entry=new Vector();
						buf.addElement(entry);
						entry.addElement(zapCodes.get(str));
						for(Enumeration c=CMClass.charClasses();c.hasMoreElements();)
						{
							CharClass C=(CharClass)c.nextElement();
							if(fromHere(V,'-',v+1,Util.padRight(C.name(),4).toUpperCase().trim()))
								entry.addElement(C.name());
						}
					}
					break;
				case 19: // -str
				case 20: // -int
				case 21: // -wis
				case 22: // -dex
				case 23: // -con
				case 24: // -cha
				case 25: // +str
				case 26: // +int
				case 27: // +wis
				case 28: // +dex
				case 29: // +con
				case 30: // +cha
					{
						val=((++v)<V.size())?Util.s_int((String)V.elementAt(v)):0;
						Vector entry=new Vector();
						buf.addElement(entry);
						entry.addElement(zapCodes.get(str));
						entry.addElement(new Integer(val));
						break;
					}
				}
			else
			{
				for(Enumeration c=CMClass.charClasses();c.hasMoreElements();)
				{
					CharClass C=(CharClass)c.nextElement();
					if(str.startsWith("-"+Util.padRight(C.name(),4).toUpperCase().trim()))
					{
						Vector entry=new Vector();
						buf.addElement(entry);
						entry.addElement(zapCodes.get("+CLASS"));
						entry.addElement(C.name());
					}
				}
				Vector cats=new Vector();
				for(Enumeration r=CMClass.races();r.hasMoreElements();)
				{
					Race R=(Race)r.nextElement();
					String cat=R.racialCategory().toUpperCase();
					if(cat.length()>6) cat=cat.substring(0,6);
					if((str.startsWith("-"+cat))&&(!cats.contains(R.racialCategory())))
					{
						Vector entry=new Vector();
						buf.addElement(entry);
						entry.addElement(zapCodes.get("+RACECAT"));
						entry.addElement(R.racialCategory());
					}
				}
				if(str.startsWith("-"+Faction.ALIGN_NAMES[Faction.ALIGN_EVIL].substring(0,3)))
				{
					Vector entry=new Vector();
					buf.addElement(entry);
					entry.addElement(zapCodes.get("+ALIGNMENT"));
					entry.addElement(Faction.ALIGN_NAMES[Faction.ALIGN_EVIL]);
				}
				if(str.startsWith("-"+Faction.ALIGN_NAMES[Faction.ALIGN_GOOD].substring(0,3)))
				{
					Vector entry=new Vector();
					buf.addElement(entry);
					entry.addElement(zapCodes.get("+ALIGNMENT"));
					entry.addElement(Faction.ALIGN_NAMES[Faction.ALIGN_GOOD]);
				}
				if(str.startsWith("-"+Faction.ALIGN_NAMES[Faction.ALIGN_NEUTRAL].substring(0,3)))
				{
					Vector entry=new Vector();
					buf.addElement(entry);
					entry.addElement(zapCodes.get("+ALIGNMENT"));
					entry.addElement(Faction.ALIGN_NAMES[Faction.ALIGN_NEUTRAL]);
				}
				if(str.startsWith("-MALE"))
				{
					Vector entry=new Vector();
					buf.addElement(entry);
					entry.addElement(zapCodes.get("+GENDER"));
					entry.addElement("M");
				}
				if(str.startsWith("-FEMALE"))
				{
					Vector entry=new Vector();
					buf.addElement(entry);
					entry.addElement(zapCodes.get("+GENDER"));
					entry.addElement("F");
				}
				if(str.startsWith("-NEUTER"))
				{
					Vector entry=new Vector();
					buf.addElement(entry);
					entry.addElement(zapCodes.get("+GENDER"));
					entry.addElement("N");
				}
				if((str.startsWith("-"))
		        &&(Factions.isRangeCodeName(str.substring(1))))
				{
					Vector entry=new Vector();
					buf.addElement(entry);
					entry.addElement(zapCodes.get("+FACTION"));
					entry.addElement(str.substring(1));
				}
				Vector entry=levelCompiledHelper(str,'-',null);
				if((entry!=null)&&(entry.size()>0))
					buf.addElement(entry);
			}
		}
		return buf;
	}

	public static boolean zapperCheckReal(Vector cset, MOB mob)
	{
		if(mob==null) return true;
		if((cset==null)||(cset.size()==0)) return true;
		getZapCodes();
		for(int c=0;c<cset.size();c++)
		{
			Vector V=(Vector)cset.elementAt(c);
			if(V.size()>0)
			switch(((Integer)V.firstElement()).intValue())
			{
			case 0: // -class
				if(!V.contains(mob.baseCharStats().getCurrentClass().name()))
					return false;
				break;
			case 1: // -baseclass
				if(!V.contains(mob.baseCharStats().getCurrentClass().baseClass()))
					return false;
				break;
			case 2: // -race
				if(!V.contains(mob.baseCharStats().getMyRace().name()))
					return false;
				break;
			case 3: // -alignment
				if(!V.contains(Sense.getAlignmentName(mob)))
					return false;
				break;
			case 4: // -gender
				if(!V.contains(""+((char)mob.baseCharStats().getStat(CharStats.GENDER))))
					return false;
				break;
			case 5: // -level
				{
					boolean found=false;
					for(int v=1;v<V.size();v+=2)
						if((v+1)<V.size())
						switch(((Integer)V.elementAt(v)).intValue())
						{
							case 37: // +lvlgr
								if((V.size()>1)&&(mob.baseEnvStats().level()>((Integer)V.elementAt(1)).intValue()))
								   found=true;
								break;
							case 38: // +lvlge
								if((V.size()>1)&&(mob.baseEnvStats().level()>=((Integer)V.elementAt(1)).intValue()))
								   found=true;
								break;
							case 39: // +lvlt
								if((V.size()>1)&&(mob.baseEnvStats().level()<((Integer)V.elementAt(1)).intValue()))
								   found=true;
								break;
							case 40: // +lvlle
								if((V.size()>1)&&(mob.baseEnvStats().level()<=((Integer)V.elementAt(1)).intValue()))
								   found=true;
								break;
							case 41: // +lvleq
								if((V.size()>1)&&(mob.baseEnvStats().level()==((Integer)V.elementAt(1)).intValue()))
								   found=true;
								break;
						}
					if(!found) return false;
				}
				break;
			case 6: // -classlevel
				{
					boolean found=false;
					int cl=mob.baseCharStats().getClassLevel(mob.baseCharStats().getCurrentClass());
					for(int v=1;v<V.size();v+=2)
						if((v+1)<V.size())
						switch(((Integer)V.elementAt(v)).intValue())
						{
							case 37: // +lvlgr
								if((V.size()>1)&&(cl>((Integer)V.elementAt(1)).intValue()))
								   found=true;
								break;
							case 38: // +lvlge
								if((V.size()>1)&&(cl>=((Integer)V.elementAt(1)).intValue()))
								   found=true;
								break;
							case 39: // +lvlt
								if((V.size()>1)&&(cl<((Integer)V.elementAt(1)).intValue()))
								   found=true;
								break;
							case 40: // +lvlle
								if((V.size()>1)&&(cl<=((Integer)V.elementAt(1)).intValue()))
								   found=true;
								break;
							case 41: // +lvleq
								if((V.size()>1)&&(cl==((Integer)V.elementAt(1)).intValue()))
								   found=true;
								break;
						}
					if(!found) return false;
				}
				break;
			case 7: // -tattoo
				{
					boolean found=false;
					for(int v=1;v<V.size();v++)
						if(mob.fetchTattoo((String)V.elementAt(v))!=null)
						{ found=true; break;}
					if(!found) return false;
				}
				break;
			case 8: // +tattoo
				{
					for(int v=1;v<V.size();v++)
						if(mob.fetchTattoo((String)V.elementAt(v))!=null)
						{ return false;}
				}
				break;
			case 9: // -name
				{
					boolean found=false;
					for(int v=1;v<V.size();v++)
						if(mob.Name().equalsIgnoreCase((String)V.elementAt(v)))
						{ found=true; break;}
					if(!found) return false;
				}
				break;
			case 10: // -player
				if(!mob.isMonster()) return false;
				break;
			case 11: // -npc
				if(mob.isMonster()) return false;
				break;
			case 12: // -racecat
				if(!V.contains(mob.baseCharStats().getMyRace().racialCategory()))
					return false;
				break;
			case 13: // +racecat
				if(V.contains(mob.baseCharStats().getMyRace().racialCategory()))
					return false;
				break;
			case 14: // -clan
				{
					if(mob.getClanID().length()==0) 
						return false;
					boolean found=false;
					for(int v=1;v<V.size();v++)
						if(mob.getClanID().equalsIgnoreCase((String)V.elementAt(v)))
						{ found=true; break;}
					if(!found) return false;
				}
				break;
			case 15: // +clan
				if(mob.getClanID().length()>0) 
					for(int v=1;v<V.size();v++)
						if(mob.getClanID().equalsIgnoreCase((String)V.elementAt(v)))
						{ return false;}
				break;
			case 44: // -deity
				{
					if(mob.getWorshipCharID().length()==0) 
						return false;
					boolean found=false;
					for(int v=1;v<V.size();v++)
						if(mob.getWorshipCharID().equalsIgnoreCase((String)V.elementAt(v)))
						{ found=true; break;}
					if(!found) return false;
				}
				break;
			case 45: // +deity
				if(mob.getWorshipCharID().length()>0) 
					for(int v=1;v<V.size();v++)
						if(mob.getWorshipCharID().equalsIgnoreCase((String)V.elementAt(v)))
						{ return false;}
				break;
			case 43: // -effects
				{
					if(mob.numEffects()==0) 
						return false;
					boolean found=false;
					for(int a=0;a<mob.numEffects();a++)
					{
						Ability A=mob.fetchEffect(a);
						if(A!=null)
						for(int v=1;v<V.size();v++)
							if(A.Name().equalsIgnoreCase((String)V.elementAt(v)))
							{ found=true; break;}
					}
					if(!found) return false;
				}
				break;
			case 46: // -faction
				{
			    	boolean found=false;
			    	for(int v=1;v<V.size();v++)
			    	    if(Factions.isFactionedThisWay(mob,(String)V.elementAt(v)))
			    	    { found=true; break;}
					if(!found) return false;
				}
				break;
			case 47: // +faction
				{
			    	for(int v=1;v<V.size();v++)
			    	    if(Factions.isFactionedThisWay(mob,(String)V.elementAt(v)))
			    	        return false;
				}
				break;
			case 42: // +effects
				if(mob.numEffects()>0) 
				for(int a=0;a<mob.numEffects();a++)
				{
					Ability A=mob.fetchEffect(a);
					if(A!=null)
					for(int v=1;v<V.size();v++)
						if(A.Name().equalsIgnoreCase((String)V.elementAt(v)))
						{ return false;}
				}
				break;
			case 16: // +name
				{
					for(int v=1;v<V.size();v++)
						if(mob.Name().equalsIgnoreCase((String)V.elementAt(v)))
						{ return false;}
				}
				break;
			case 17: // -anyclass
				{
					boolean found=false;
					for(int v=1;v<V.size();v++)
						if(mob.baseCharStats().getClassLevel((String)V.elementAt(v))>=0)
						{ found=true; break;}
					if(!found) return false;
				}
				break;
			case 18: // +anyclass
				for(int v=1;v<V.size();v++)
					if(mob.baseCharStats().getClassLevel((String)V.elementAt(v))>=0)
					{ return false;}
				break;
			case 19: // +str
				if((V.size()>1)&&(mob.baseCharStats().getStat(CharStats.STRENGTH)>(((Integer)V.elementAt(1)).intValue())))
				   return false;
				break;
			case 20: // +int
				if((V.size()>1)&&(mob.baseCharStats().getStat(CharStats.INTELLIGENCE)>(((Integer)V.elementAt(1)).intValue())))
				   return false;
				break;
			case 21: // +wis
				if((V.size()>1)&&(mob.baseCharStats().getStat(CharStats.WISDOM)>(((Integer)V.elementAt(1)).intValue())))
				   return false;
				break;
			case 22: // +dex
				if((V.size()>1)&&(mob.baseCharStats().getStat(CharStats.DEXTERITY)>(((Integer)V.elementAt(1)).intValue())))
				   return false;
				break;
			case 23: // +con
				if((V.size()>1)&&(mob.baseCharStats().getStat(CharStats.CONSTITUTION)>(((Integer)V.elementAt(1)).intValue())))
				   return false;
				break;
			case 24: // +cha
				if((V.size()>1)&&(mob.baseCharStats().getStat(CharStats.CHARISMA)>(((Integer)V.elementAt(1)).intValue())))
				   return false;
				break;
			case 25: // -str
				if((V.size()>1)&&(mob.baseCharStats().getStat(CharStats.STRENGTH)<(((Integer)V.elementAt(1)).intValue())))
				   return false;
				break;
			case 26: // -int
				if((V.size()>1)&&(mob.baseCharStats().getStat(CharStats.INTELLIGENCE)<(((Integer)V.elementAt(1)).intValue())))
				   return false;
				break;
			case 27: // -wis
				if((V.size()>1)&&(mob.baseCharStats().getStat(CharStats.WISDOM)<(((Integer)V.elementAt(1)).intValue())))
				   return false;
				break;
			case 28: // -dex
				if((V.size()>1)&&(mob.baseCharStats().getStat(CharStats.DEXTERITY)<(((Integer)V.elementAt(1)).intValue())))
				   return false;
				break;
			case 29: // -con
				if((V.size()>1)&&(mob.baseCharStats().getStat(CharStats.CONSTITUTION)<(((Integer)V.elementAt(1)).intValue())))
				   return false;
				break;
			case 30: // -cha
				if((V.size()>1)&&(mob.baseCharStats().getStat(CharStats.CHARISMA)<(((Integer)V.elementAt(1)).intValue())))
				   return false;
				break;
			case 31: // -area
				{
					boolean found=false;
					if(mob.location()!=null)
						for(int v=1;v<V.size();v++)
							if(mob.location().getArea().Name().equalsIgnoreCase((String)V.elementAt(v)))
							{ found=true; break;}
					if(!found) return false;
				}
				break;
			case 32: // +area
				if(mob.location()!=null)
					for(int v=1;v<V.size();v++)
						if(mob.location().getArea().Name().equalsIgnoreCase((String)V.elementAt(v)))
						{ return false;}
				break;
			case 33: // +item
				{
					boolean found=false;
					for(int v=1;v<V.size();v++)
						if(mob.fetchInventory((String)V.elementAt(v))!=null)
						{ found=true; break;}
					if(!found) return false;
				}
				break;
			case 34: // +class
				if(V.contains(mob.baseCharStats().getCurrentClass().name()))
					return false;
				break;
			case 35: // +alignment
				if(V.contains(Sense.getAlignmentName(mob)))
					return false;
				break;
			case 36: // +gender
				if(V.contains(""+((char)mob.baseCharStats().getStat(CharStats.GENDER))))
					return false;
				break;
			case 37: // +lvlgr
				if((V.size()>1)&&(mob.baseEnvStats().level()>((Integer)V.elementAt(1)).intValue()))
				   return false;
				break;
			case 38: // +lvlge
				if((V.size()>1)&&(mob.baseEnvStats().level()>=((Integer)V.elementAt(1)).intValue()))
				   return false;
				break;
			case 39: // +lvlt
				if((V.size()>1)&&(mob.baseEnvStats().level()<((Integer)V.elementAt(1)).intValue()))
				   return false;
				break;
			case 40: // +lvlle
				if((V.size()>1)&&(mob.baseEnvStats().level()<=((Integer)V.elementAt(1)).intValue()))
				   return false;
				break;
			case 41: // +lvleq
				if((V.size()>1)&&(mob.baseEnvStats().level()==((Integer)V.elementAt(1)).intValue()))
				   return false;
				break;
			}
		}
		return true;
	}
	
	public static boolean zapperCheck(String text, MOB mob)
	{
		if(mob==null) return true;
		if(mob.charStats()==null) return true;
		if(text.trim().length()==0) return true;
		getZapCodes();

		String mobClass=Util.padRight(mob.charStats().displayClassName(),4).toUpperCase().trim();
		String mobRaceCat=mob.charStats().getMyRace().racialCategory().toUpperCase();
		if(!mob.charStats().getMyRace().name().equals(mob.charStats().raceName()))
		{
			Race R=CMClass.getRace(mob.charStats().raceName());
			if(R!=null) mobRaceCat=R.racialCategory().toUpperCase();
			else mobRaceCat=mob.charStats().raceName().toUpperCase();
		}
		if(mobRaceCat.length()>6) mobRaceCat=mobRaceCat.substring(0,6);
		String mobRace=mob.charStats().raceName().toUpperCase();
		if(mobRace.length()>6) mobRace=mobRace.substring(0,6);
		String mobAlign=Sense.getAlignmentName(mob).substring(0,3);
		String mobGender=mob.charStats().genderName().toUpperCase();
		int level=mob.envStats().level();
		int classLevel=mob.charStats().getClassLevel(mob.charStats().getCurrentClass());

		Vector V=Util.parse(text.toUpperCase());
		if(CMSecurity.isASysOp(mob)
		||((mob.location()!=null)&&(mob.location().getArea().amISubOp(mob.Name()))))
			for(int v=0;v<V.size();v++)
			{
				String str=(String)V.elementAt(v);
				if(str.equals("+SYSOP")) return true;
				else
				if(str.equals("-SYSOP")) return false;
			}
		for(int v=0;v<V.size();v++)
		{
			String str=(String)V.elementAt(v);
			int val=-1;
			if(zapCodes.containsKey(str))
				switch(((Integer)zapCodes.get(str)).intValue())
				{
				case 0: // -class
					if(!fromHere(V,'+',v+1,mobClass)) return false;
					break;
				case 1: // -baseclass
					if((!fromHere(V,'+',v+1,Util.padRight(mob.charStats().getCurrentClass().baseClass(),4).toUpperCase().trim()))
					&&(!fromHere(V,'+',v+1,mobClass))) return false;
					break;
				case 2: // -Race
					if(!fromHere(V,'+',v+1,mobRace))
						return false;
					break;
				case 3: // -Alignment
					if(!fromHere(V,'+',v+1,mobAlign)) return false;
					break;
				case 4: // -Gender
					if(!fromHere(V,'+',v+1,mobGender)) return false;
					break;
				case 5: // -Levels
					if(!levelCheck(Util.combine(V,v+1),'+',0,level)) return false;
					break;
				case 6: // -ClassLevels
					if(!levelCheck(Util.combine(V,v+1),'+',0,classLevel)) return false;
					break;
				case 7: // -tattoos
					if(!tattooCheck(V,'+',v+1,mob)) return false;
					break;
				case 8: // +tattoos
					if(tattooCheck(V,'-',v+1,mob)) return false;
					break;
				case 9: // -names
					if(!nameCheck(V,'+',v+1,mob)) return false;
					break;
				case 10: // -Player
					if(!mob.isMonster()) return false;
					break;
				case 11: // -MOB
					if(mob.isMonster()) return false;
					break;
				case 12: // -Racecat
					if(!fromHere(V,'+',v+1,mobRaceCat))
						return false;
					break;
				case 13: // +Racecat
					if(fromHere(V,'-',v+1,mobRaceCat))
						return false;
					break;
				case 14: // -Clan
					if((mob.getClanID().length()==0)
					||(!fromHere(V,'+',v+1,mob.getClanID().toUpperCase())))
						return false;
					break;
				case 15: // +Clan
					if((mob.getClanID().length()>0)
					&&(fromHere(V,'-',v+1,mob.getClanID().toUpperCase())))
						return false;
					break;
				case 44: // -Deity
					if((mob.getWorshipCharID().length()==0)
					||(!fromHere(V,'+',v+1,mob.getWorshipCharID().toUpperCase())))
						return false;
					break;
				case 45: // +Deity
					if((mob.getWorshipCharID().length()>0)
					&&(fromHere(V,'-',v+1,mob.getWorshipCharID().toUpperCase())))
						return false;
					break;
				case 16: // +names
					if(nameCheck(V,'-',v+1,mob))
						return false;
					break;
				case 17: // -anyclass
					{
						boolean found=false;
						for(int c=0;c<mob.charStats().numClasses();c++)
							if(fromHere(V,'+',v+1,Util.padRight(mob.charStats().getMyClass(c).name(),4).toUpperCase().trim()))
								found=true;
						if(!found) return false;
					}
					break;
				case 18: // +anyclass
					for(int c=0;c<mob.charStats().numClasses();c++)
						if(fromHere(V,'-',v+1,Util.padRight(mob.charStats().getMyClass(c).name(),4).toUpperCase().trim()))
							return false;
					break;
				case 19: // -str
					val=((++v)<V.size())?Util.s_int((String)V.elementAt(v)):0;
					if(mob.charStats().getStat(CharStats.STRENGTH)<val)
						return false;
					break;
				case 20: // -int
					val=((++v)<V.size())?Util.s_int((String)V.elementAt(v)):0;
					if(mob.charStats().getStat(CharStats.INTELLIGENCE)<val)
						return false;
					break;
				case 21: // -wis
					val=((++v)<V.size())?Util.s_int((String)V.elementAt(v)):0;
					if(mob.charStats().getStat(CharStats.WISDOM)<val)
						return false;
					break;
				case 22: // -dex
					val=((++v)<V.size())?Util.s_int((String)V.elementAt(v)):0;
					if(mob.charStats().getStat(CharStats.DEXTERITY)<val)
						return false;
					break;
				case 23: // -con
					val=((++v)<V.size())?Util.s_int((String)V.elementAt(v)):0;
					if(mob.charStats().getStat(CharStats.CONSTITUTION)<val)
						return false;
					break;
				case 24: // -cha
					val=((++v)<V.size())?Util.s_int((String)V.elementAt(v)):0;
					if(mob.charStats().getStat(CharStats.CHARISMA)<val)
						return false;
					break;
				case 25: // +str
					val=((++v)<V.size())?Util.s_int((String)V.elementAt(v)):0;
					if(mob.charStats().getStat(CharStats.STRENGTH)>val)
						return false;
					break;
				case 26: // +int
					val=((++v)<V.size())?Util.s_int((String)V.elementAt(v)):0;
					if(mob.charStats().getStat(CharStats.INTELLIGENCE)>val)
						return false;
					break;
				case 27: // +wis
					val=((++v)<V.size())?Util.s_int((String)V.elementAt(v)):0;
					if(mob.charStats().getStat(CharStats.WISDOM)>val)
						return false;
					break;
				case 28: // +dex
					val=((++v)<V.size())?Util.s_int((String)V.elementAt(v)):0;
					if(mob.charStats().getStat(CharStats.DEXTERITY)>val)
						return false;
					break;
				case 29: // +con
					val=((++v)<V.size())?Util.s_int((String)V.elementAt(v)):0;
					if(mob.charStats().getStat(CharStats.CONSTITUTION)>val)
						return false;
					break;
				case 30: // +cha
					val=((++v)<V.size())?Util.s_int((String)V.elementAt(v)):0;
					if(mob.charStats().getStat(CharStats.CHARISMA)>val)
						return false;
					break;
				case 31: // +area
					if(areaCheck(V,'-',v+1,mob))
						return false;
					break;
				case 32: // -area
					if(!areaCheck(V,'+',v+1,mob)) return false;
					break;
				case 33: // +item
					if(!itemCheck(V,'+',v+1,mob)) return false;
					break;
				case 43: // -Effects
				{
					if(mob.numEffects()==0) return false;
					boolean found=false;
					for(int a=0;a<mob.numEffects();a++)
					{
						Ability A=mob.fetchEffect(a);
						if((A!=null)&&(fromHere(V,'+',v+1,A.Name().toUpperCase())))
						{ found=true; break;}
					}
					if(!found) return false;
					break;
				}
				case 42: // +Effects
					for(int a=0;a<mob.numEffects();a++)
					{
						Ability A=mob.fetchEffect(a);
						if((A!=null)&&(fromHere(V,'-',v+1,A.Name().toUpperCase())))
							return false;
					}
					break;
				case 46: // -Faction
					if(!factionCheck(V,'+',v+1,mob)) 
					    return false;
					break;
				}
			else
			if(str.startsWith("-"+mobClass)) return false;
			else
			if(str.startsWith("-"+mobRace)) return false;
			else
			if(str.startsWith("-"+mobAlign)) return false;
			else
			if(str.startsWith("-"+mobGender)) return false;
			else
			if(levelCheck(str,'-',0,level)) return false;
			else
			if(str.startsWith("-")
			&&(Factions.isFactionedThisWay(mob,str.substring(1))))
				return false;
		}
		return true;
	}

}
